<?php

/*
Plugin Name: VamTam Importers
Description: This plugin is used in order to import the sample content for VamTam themes
Version: 2.15.0
Author: VamTam
Author URI: http://vamtam.com
*/

class Vamtam_Importers {
	public static $regenerate_thumbnails;

	public function __construct() {
		add_action( 'admin_init', array( __CLASS__, 'admin_init' ), 1 );
		add_action( 'plugins_loaded', array( __CLASS__, 'plugins_loaded' ) );

		if ( ! class_exists( 'Vamtam_Updates_2' ) ) {
			require 'vamtam-updates/class-vamtam-updates.php';
		}

		new Vamtam_Updates_2( __FILE__ );
	}

	public static function admin_init() {
		add_action( 'vamtam_before_content_import', array( __CLASS__, 'before_content_import' ) );
		add_action( 'vamtam_after_content_import', array( __CLASS__, 'after_content_import' ) );

		require 'importers/importer/importer.php';
		require 'importers/widget-importer/importer.php';
		require 'importers/revslider/importer.php';
		require 'importers/ninja-forms/importer.php';
		require 'importers/acx-coming-soon/importer.php';
		require 'importers/booked/importer.php';
		require 'importers/google-maps-easy/importer.php';
	}

	public static function plugins_loaded() {
		include 'wp-background-process/wp-async-request.php';
		include 'wp-background-process/wp-background-process.php';
		include 'vamtam-regenerate-thumbnails.php';

		self::$regenerate_thumbnails = new Vamtam_Regenerate_Thumbnails();
	}

	/**
	 * Initialize thumbnail generation
	 */
	protected static function process_thumbnails() {
		$attachment_ids = get_option( 'vamtam_last_import_map' )['attachments'];

		foreach ( $attachment_ids as $attachment_id ) {
			self::$regenerate_thumbnails->push_to_queue( $attachment_id );
		}

		self::$regenerate_thumbnails->save()->dispatch();
	}

	public static function before_content_import() {
		wp_suspend_cache_invalidation( true );

		self::generic_option_import( 'jetpack', array( __CLASS__, 'jetpack_import' ) );
		self::generic_option_import( 'foodpress', array( __CLASS__, 'foodpress_import' ) );
		self::generic_option_import( 'the-events-calendar', array( __CLASS__, 'tribe_events_import' ) );

		if ( class_exists( 'FLBuilderModel' ) ) {
			$post_types = get_post_types( array( 'public' => true ), 'objects' );
			$post_types = apply_filters( 'fl_builder_admin_settings_post_types', $post_types );
			$post_types = array_diff( array_keys( $post_types ), array( 'attachment', 'fl-builder-template' ) );

			FLBuilderModel::update_admin_settings_option( '_fl_builder_post_types', $post_types, true );

			FLBuilderModel::save_global_settings( json_decode( file_get_contents( VAMTAM_SAMPLES_DIR . 'beaver-global-settings.json' ), true ) );

			FLBuilderUserAccess::save_settings( json_decode( file_get_contents( VAMTAM_SAMPLES_DIR . 'beaver-user-access.json' ), true ) );
		}

		wp_suspend_cache_invalidation( false );

		require 'vamtam-import-search-replace.php';
	}

	public static function after_content_import() {
		$map = get_option( 'vamtam_last_import_map' );

		$posts = get_posts( array(
			'post_type'      => get_post_types(),
			'posts_per_page' => -1,
			'meta_query'     => array(
				'key'     => '_fl_builder_data',
				'compare' => 'EXISTS',
			),
			'orderby' => 'ID',
			'order' => 'ASC',
		) );

		// loop through the Beaver data for all pages and map old post/term IDs to the new ones (after import)
		foreach ( $posts as $post ) {
			$data = get_post_meta( $post->ID, '_fl_builder_data', true );

			if ( ! $data ) {
				$meta = get_post_meta( $post->ID );

				if ( isset( $meta[ '_fl_builder_data' ] ) ) {
					$data = maybe_unserialize( self::fix_serialized( $meta[ '_fl_builder_data' ][0] ) );
				} else {
					echo "missing _fl_builder_data for {$post->ID} {$post->post_type}\n";
					unset( $data );
				}
			}

			if ( isset( $data ) && is_array( $data ) ) {
				foreach ( $data as $node_id => $node ) {
					// only for vamtam-blog and vamtam-projects modules
					if ( $node->type === 'module' && in_array( $node->settings->type, array( 'vamtam-blog', 'vamtam-projects' ), true ) ) {
						$settings = get_object_vars( $node->settings );

						foreach ( $settings as $setting_name => $setting_value ) {
							if ( $setting_name !== 'posts_per_page' ) {
								if ( strpos( $setting_name, 'posts_' ) === 0 && ! empty( $setting_value ) ) {
									$setting_value = explode( ',', $setting_value );

									foreach ( $setting_value as &$post_id ) {
										$post_id = $map['posts'][ (int)$post_id ];
									}

									$data[ $node_id ]->settings->{$setting_name} = implode( ',', $setting_value );
								} elseif ( strpos( $setting_name, 'tax_' ) === 0 && ! empty( $setting_value ) ) {
									$setting_value = explode( ',', $setting_value );

									foreach ( $setting_value as &$term_id ) {
										$term_id = $map['terms'][ (int)$term_id ];
									}

									$data[ $node_id ]->settings->{$setting_name} = implode( ',', $setting_value );
								}
							}
						}
					} else if ( $node->type === 'module' && $node->settings->type === 'widget' && isset( $node->settings->{"widget-nav_menu"} ) ) {
						$from = $node->settings->{'widget-nav_menu'}->nav_menu;

						if ( isset( $map['terms'][ (int) $from ] ) ) {
							$mapped = $map['terms'][ (int) $from ];
							echo "will map menu {$from} to {$mapped}\n";
							$data[ $node_id ]->settings->{"widget-nav_menu"}->nav_menu = $mapped;
						} else {
							echo "no mapped id for nav_menu {$from}\n";
						}
					}
				}

				update_post_meta( $post->ID, '_fl_builder_data', $data );
			}
		}

		self::process_thumbnails();

		self::megamenu_import();
	}

	public static function set_menu_locations() {
		$map  = get_option( 'vamtam_last_import_map', false );
		$path = VAMTAM_SAMPLES_DIR . 'theme-mods.json';

		if ( $map && ! get_theme_mod( 'vamtam_force_demo_menu', false ) && file_exists( $path ) ) {
			$theme_mods = json_decode( file_get_contents( $path ), true );

			if ( isset( $theme_mods['nav_menu_locations'] ) ) {
				foreach ( $theme_mods['nav_menu_locations'] as $location => $term_id ) {
					if ( isset( $map['terms'][ (int)$term_id ] ) ) {
						$theme_mods['nav_menu_locations'][ $location ] = $map['terms'][ (int)$term_id ];
					}
				}
			}

			foreach ( $theme_mods as $opt_name => $mod_val ) {
				set_theme_mod( $opt_name, $mod_val );
			}

			set_theme_mod( 'vamtam_force_demo_menu', true );
		}
	}

	public static function generic_option_import( $file, $callback ) {
		$path = VAMTAM_SAMPLES_DIR . $file . '.json';

		if ( file_exists( $path ) ) {
			$settings = json_decode( file_get_contents( $path ), true );

			foreach ( $settings as $opt_name => $opt_val ) {
				update_option( $opt_name, $opt_val );
			}

			call_user_func( $callback );
		}
	}

	public static function megamenu_import() {
		$file = VAMTAM_SAMPLES_DIR . 'megamenu.json';

		if ( file_exists( $file ) ) {
			$exported = json_decode( file_get_contents( $file ), true );

			update_site_option( 'megamenu_themes', $exported['themes'] );
			update_site_option( 'megamenu_settings', $exported['settings'] );

			do_action( 'megamenu_after_theme_import' );
		}
	}

	public static function jetpack_import() {
		Jetpack::load_modules();

		if ( class_exists( 'Jetpack_Portfolio' ) ) {
			Jetpack_Portfolio::init()->register_post_types();
		}

		if ( class_exists( 'Jetpack_Testimonial' ) ) {
			Jetpack_Testimonial::init()->register_post_types();
		}
	}

	public static function foodpress_import() {
		if ( function_exists( 'foodpress_generate_options_css' ) ) {
			foodpress_generate_options_css();
		}
	}

	public static function tribe_events_import() {
		// no cache to regenerate at this time
	}

	/**
	 * @return string
	 */
	static public function fix_serialized( $src ) {
		if ( empty( $src ) ) {
			return $src;
		}

		$data = maybe_unserialize( $src );

		// return if maybe_unserialize() returns an object or array, this is good.
		if( is_object( $data ) || is_array( $data ) ) {
			return $data;
		}

		$data = preg_replace_callback( '!s:(\d+):([\\\\]?"[\\\\]?"|[\\\\]?"((.*?)[^\\\\])[\\\\]?");!', array( __CLASS__, 'fix_serial_callback' ), $src );

		if ( ! isset( $data ) && strlen( $data ) === 0 ) {
			return $src;
		}

		return $data;
	}

	/**
	 * @return string
	 */
	static public function fix_serial_callback( $matches ) {
		if ( ! isset( $matches[3] ) ) {
			return $matches[0];
		}

		return 's:' . strlen( self::unescape_mysql( $matches[3] ) ) . ':"' . self::unescape_quotes( $matches[3] ) . '";';
	}

	/**
	 * Unescape to avoid dump-text issues.
	 *
	 * @access private
	 * @return string
	 */
	static private function unescape_mysql( $value ) {
		return str_replace( array( "\\\\", "\\0", "\\n", "\\r", "\Z", "\'", '\"' ),
			array( "\\",   "\0",  "\n",  "\r",  "\x1a", "'", '"' ),
		$value );
	}

	/**
	 * Fix strange behaviour if you have escaped quotes in your replacement.
	 *
	 * @access private
	 * @return string
	 */
	static private function unescape_quotes( $value ) {
		return str_replace( '\"', '"', $value );
	}
}

new Vamtam_Importers;
